<?php
/**
 *
 * @Packge      Tourm 
 * @Author      Themeholy
 * @Author URL  https://themeforest.net/user/themeholy 
 * @version     2.1.0
 *
 */

/**
 * Enqueue style of child theme
 */
function tourm_child_enqueue_styles() {
    wp_enqueue_style( 'tourm-style', get_template_directory_uri() . '/style.css' );
    wp_enqueue_style( 'tourm-child-style', get_stylesheet_directory_uri() . '/style.css',array( 'tourm-style' ),wp_get_theme()->get('Version'));
    
    // Enqueue room booking styles
    wp_enqueue_style( 'room-booking-style', get_stylesheet_directory_uri() . '/assets/css/room-booking.css', array(), '1.0.0' );
    
    // Enqueue Flatpickr for date picker
    wp_enqueue_style( 'flatpickr-css', 'https://cdn.jsdelivr.net/npm/flatpickr/dist/flatpickr.min.css', array(), '4.6.13' );
    wp_enqueue_script( 'flatpickr-js', 'https://cdn.jsdelivr.net/npm/flatpickr/dist/flatpickr.min.js', array('jquery'), '4.6.13', true );
    
    // Enqueue room booking scripts
    wp_enqueue_script( 'room-booking-js', get_stylesheet_directory_uri() . '/assets/js/room-booking.js', array('jquery', 'flatpickr-js'), '1.0.0', true );
    
    // Localize script for AJAX
    wp_localize_script( 'room-booking-js', 'roomBooking', array(
        'ajaxurl' => admin_url( 'admin-ajax.php' ),
        'nonce' => wp_create_nonce( 'room_booking_nonce' ),
    ));
}
add_action( 'wp_enqueue_scripts', 'tourm_child_enqueue_styles', 100000 );

// ============================================
// 1. CHANGE PRODUCT SLUG TO /rooms/
// ============================================

function change_product_slug_to_rooms( $args, $post_type ) {
    if ( 'product' === $post_type ) {
        $args['rewrite']['slug'] = 'rooms';
        $args['has_archive'] = true;
    }
    return $args;
}
add_filter( 'register_post_type_args', 'change_product_slug_to_rooms', 10, 2 );

// Flush rewrite rules on theme activation
function flush_rewrite_rules_on_activation() {
    change_product_slug_to_rooms( array(), 'product' );
    flush_rewrite_rules();
}
add_action( 'after_switch_theme', 'flush_rewrite_rules_on_activation' );

// ============================================
// 2. RENAME PRODUCT TERMINOLOGY TO ROOMS
// ============================================

function rename_product_labels( $labels ) {
    $labels->name = 'Rooms';
    $labels->singular_name = 'Room';
    $labels->add_new = 'Add New Room';
    $labels->add_new_item = 'Add New Room';
    $labels->edit_item = 'Edit Room';
    $labels->new_item = 'New Room';
    $labels->view_item = 'View Room';
    $labels->search_items = 'Search Rooms';
    $labels->not_found = 'No rooms found';
    $labels->not_found_in_trash = 'No rooms found in Trash';
    $labels->all_items = 'All Rooms';
    $labels->menu_name = 'Rooms';
    return $labels;
}
add_filter( 'post_type_labels_product', 'rename_product_labels' );

// Rename product category to Room Categories
function rename_product_category_labels( $labels ) {
    $labels->name = 'Room Categories';
    $labels->singular_name = 'Room Category';
    $labels->menu_name = 'Room Categories';
    $labels->all_items = 'All Room Categories';
    $labels->edit_item = 'Edit Room Category';
    $labels->view_item = 'View Room Category';
    $labels->update_item = 'Update Room Category';
    $labels->add_new_item = 'Add New Room Category';
    $labels->new_item_name = 'New Room Category Name';
    $labels->search_items = 'Search Room Categories';
    return $labels;
}
add_filter( 'taxonomy_labels_product_cat', 'rename_product_category_labels' );

// Rename product attributes to Room Attributes
function rename_product_attribute_labels( $labels ) {
    $labels->name = 'Room Attributes';
    $labels->singular_name = 'Room Attribute';
    $labels->menu_name = 'Room Attributes';
    return $labels;
}
add_filter( 'woocommerce_attribute_taxonomy_labels', 'rename_product_attribute_labels' );

// ============================================
// 3. ADD CUSTOM FIELDS FOR ROOM DATA
// ============================================

function add_room_custom_fields() {
    // Room Details Meta Box
    add_meta_box(
        'room_details_meta_box',
        'Room Details',
        'room_details_meta_box_callback',
        'product',
        'normal',
        'high'
    );
    
    // Room Pricing Meta Box
    add_meta_box(
        'room_pricing_meta_box',
        'Room Pricing Settings',
        'room_pricing_meta_box_callback',
        'product',
        'normal',
        'high'
    );
    
    // Room Availability Meta Box
    add_meta_box(
        'room_availability_meta_box',
        'Room Availability Settings',
        'room_availability_meta_box_callback',
        'product',
        'normal',
        'high'
    );
    
    // Extra Services Meta Box
    add_meta_box(
        'room_extra_services_meta_box',
        'Extra Services',
        'room_extra_services_meta_box_callback',
        'product',
        'normal',
        'default'
    );
}
add_action( 'add_meta_boxes', 'add_room_custom_fields' );

// Room Details Meta Box Callback
function room_details_meta_box_callback( $post ) {
    wp_nonce_field( 'room_details_meta_box', 'room_details_meta_box_nonce' );
    
    $room_size = get_post_meta( $post->ID, '_room_size', true );
    $room_occupancy = get_post_meta( $post->ID, '_room_occupancy', true );
    $room_max_guests = get_post_meta( $post->ID, '_room_max_guests', true );
    $room_type = get_post_meta( $post->ID, '_room_type', true );
    $room_view = get_post_meta( $post->ID, '_room_view', true );
    $room_bed_type = get_post_meta( $post->ID, '_room_bed_type', true );
    $room_amenities = get_post_meta( $post->ID, '_room_amenities', true );
    if ( !is_array( $room_amenities ) ) {
        $room_amenities = array();
    }
    
    $available_amenities = array(
        'wifi' => 'WiFi',
        'tv' => 'TV',
        'ac' => 'Air Conditioning',
        'minibar' => 'Minibar',
        'safe' => 'Safe',
        'balcony' => 'Balcony',
        'ocean_view' => 'Ocean View',
        'room_service' => 'Room Service',
        'breakfast' => 'Breakfast Included',
        'parking' => 'Parking',
        'pool_access' => 'Pool Access',
        'gym_access' => 'Gym Access',
        'spa' => 'Spa Access',
        'kitchen' => 'Kitchen',
        'washing_machine' => 'Washing Machine',
    );
    
    ?>
    <table class="form-table">
        <tr>
            <th><label for="room_size">Room Size (sq ft)</label></th>
            <td><input type="number" id="room_size" name="room_size" value="<?php echo esc_attr( $room_size ); ?>" /></td>
        </tr>
        <tr>
            <th><label for="room_occupancy">Occupancy</label></th>
            <td><input type="text" id="room_occupancy" name="room_occupancy" value="<?php echo esc_attr( $room_occupancy ); ?>" placeholder="e.g., 2 Adults, 1 Child" /></td>
        </tr>
        <tr>
            <th><label for="room_max_guests">Maximum Guests</label></th>
            <td><input type="number" id="room_max_guests" name="room_max_guests" value="<?php echo esc_attr( $room_max_guests ); ?>" min="1" /></td>
        </tr>
        <tr>
            <th><label for="room_type">Room Type</label></th>
            <td>
                <select id="room_type" name="room_type">
                    <option value="">Select Type</option>
                    <option value="single" <?php selected( $room_type, 'single' ); ?>>Single</option>
                    <option value="double" <?php selected( $room_type, 'double' ); ?>>Double</option>
                    <option value="twin" <?php selected( $room_type, 'twin' ); ?>>Twin</option>
                    <option value="suite" <?php selected( $room_type, 'suite' ); ?>>Suite</option>
                    <option value="deluxe" <?php selected( $room_type, 'deluxe' ); ?>>Deluxe</option>
                    <option value="presidential" <?php selected( $room_type, 'presidential' ); ?>>Presidential</option>
                </select>
            </td>
        </tr>
        <tr>
            <th><label for="room_view">View</label></th>
            <td>
                <select id="room_view" name="room_view">
                    <option value="">Select View</option>
                    <option value="ocean" <?php selected( $room_view, 'ocean' ); ?>>Ocean View</option>
                    <option value="garden" <?php selected( $room_view, 'garden' ); ?>>Garden View</option>
                    <option value="city" <?php selected( $room_view, 'city' ); ?>>City View</option>
                    <option value="mountain" <?php selected( $room_view, 'mountain' ); ?>>Mountain View</option>
                    <option value="pool" <?php selected( $room_view, 'pool' ); ?>>Pool View</option>
                </select>
            </td>
        </tr>
        <tr>
            <th><label for="room_bed_type">Bed Type</label></th>
            <td>
                <select id="room_bed_type" name="room_bed_type">
                    <option value="">Select Bed Type</option>
                    <option value="single" <?php selected( $room_bed_type, 'single' ); ?>>Single Bed</option>
                    <option value="double" <?php selected( $room_bed_type, 'double' ); ?>>Double Bed</option>
                    <option value="queen" <?php selected( $room_bed_type, 'queen' ); ?>>Queen Bed</option>
                    <option value="king" <?php selected( $room_bed_type, 'king' ); ?>>King Bed</option>
                    <option value="twin" <?php selected( $room_bed_type, 'twin' ); ?>>Twin Beds</option>
                </select>
            </td>
        </tr>
        <tr>
            <th><label>Amenities</label></th>
            <td>
                <?php foreach ( $available_amenities as $key => $label ) : ?>
                    <label style="display: block; margin-bottom: 5px;">
                        <input type="checkbox" name="room_amenities[]" value="<?php echo esc_attr( $key ); ?>" <?php checked( in_array( $key, $room_amenities ) ); ?> />
                        <?php echo esc_html( $label ); ?>
                    </label>
                <?php endforeach; ?>
            </td>
        </tr>
    </table>
    <?php
}

// Room Pricing Meta Box Callback
function room_pricing_meta_box_callback( $post ) {
    wp_nonce_field( 'room_pricing_meta_box', 'room_pricing_meta_box_nonce' );
    
    $pricing_type = get_post_meta( $post->ID, '_pricing_type', true );
    $price_per_night = get_post_meta( $post->ID, '_price_per_night', true );
    $price_per_person = get_post_meta( $post->ID, '_price_per_person', true );
    $deposit_percentage = get_post_meta( $post->ID, '_deposit_percentage', true );
    if ( empty( $deposit_percentage ) ) {
        $deposit_percentage = 30; // Default 30%
    }
    
    ?>
    <table class="form-table">
        <tr>
            <th><label for="pricing_type">Pricing Type</label></th>
            <td>
                <select id="pricing_type" name="pricing_type">
                    <option value="per_night" <?php selected( $pricing_type, 'per_night' ); ?>>Per Night</option>
                    <option value="per_person" <?php selected( $pricing_type, 'per_person' ); ?>>Per Person</option>
                    <option value="per_night_per_person" <?php selected( $pricing_type, 'per_night_per_person' ); ?>>Per Night Per Person</option>
                </select>
            </td>
        </tr>
        <tr>
            <th><label for="price_per_night">Price Per Night</label></th>
            <td><input type="number" id="price_per_night" name="price_per_night" value="<?php echo esc_attr( $price_per_night ); ?>" step="0.01" /></td>
        </tr>
        <tr>
            <th><label for="price_per_person">Price Per Person (if applicable)</label></th>
            <td><input type="number" id="price_per_person" name="price_per_person" value="<?php echo esc_attr( $price_per_person ); ?>" step="0.01" /></td>
        </tr>
        <tr>
            <th><label for="deposit_percentage">Deposit Percentage (%)</label></th>
            <td><input type="number" id="deposit_percentage" name="deposit_percentage" value="<?php echo esc_attr( $deposit_percentage ); ?>" min="0" max="100" /></td>
        </tr>
    </table>
    <?php
}

// Room Availability Meta Box Callback
function room_availability_meta_box_callback( $post ) {
    wp_nonce_field( 'room_availability_meta_box', 'room_availability_meta_box_nonce' );
    
    $min_stay = get_post_meta( $post->ID, '_min_stay', true );
    $max_stay = get_post_meta( $post->ID, '_max_stay', true );
    $check_in_time = get_post_meta( $post->ID, '_check_in_time', true );
    $check_out_time = get_post_meta( $post->ID, '_check_out_time', true );
    
    ?>
    <table class="form-table">
        <tr>
            <th><label for="min_stay">Minimum Stay (nights)</label></th>
            <td><input type="number" id="min_stay" name="min_stay" value="<?php echo esc_attr( $min_stay ); ?>" min="1" /></td>
        </tr>
        <tr>
            <th><label for="max_stay">Maximum Stay (nights)</label></th>
            <td><input type="number" id="max_stay" name="max_stay" value="<?php echo esc_attr( $max_stay ); ?>" min="1" /></td>
        </tr>
        <tr>
            <th><label for="check_in_time">Check-in Time</label></th>
            <td><input type="time" id="check_in_time" name="check_in_time" value="<?php echo esc_attr( $check_in_time ? $check_in_time : '14:00' ); ?>" /></td>
        </tr>
        <tr>
            <th><label for="check_out_time">Check-out Time</label></th>
            <td><input type="time" id="check_out_time" name="check_out_time" value="<?php echo esc_attr( $check_out_time ? $check_out_time : '11:00' ); ?>" /></td>
        </tr>
    </table>
    <?php
}

// Extra Services Meta Box Callback
function room_extra_services_meta_box_callback( $post ) {
    wp_nonce_field( 'room_extra_services_meta_box', 'room_extra_services_meta_box_nonce' );
    
    $extra_services = get_post_meta( $post->ID, '_extra_services', true );
    if ( !is_array( $extra_services ) ) {
        $extra_services = array();
    }
    
    ?>
    <div id="extra-services-container">
        <p><button type="button" class="button" id="add-extra-service">Add Extra Service</button></p>
        <div id="extra-services-list">
            <?php if ( !empty( $extra_services ) ) : ?>
                <?php foreach ( $extra_services as $index => $service ) : ?>
                    <div class="extra-service-item" style="margin-bottom: 10px; padding: 10px; border: 1px solid #ddd;">
                        <input type="text" name="extra_services[<?php echo $index; ?>][name]" value="<?php echo esc_attr( $service['name'] ); ?>" placeholder="Service Name" style="width: 40%; margin-right: 10px;" />
                        <input type="number" name="extra_services[<?php echo $index; ?>][price]" value="<?php echo esc_attr( $service['price'] ); ?>" placeholder="Price" step="0.01" style="width: 20%; margin-right: 10px;" />
                        <select name="extra_services[<?php echo $index; ?>][type]" style="width: 20%; margin-right: 10px;">
                            <option value="per_night" <?php selected( $service['type'], 'per_night' ); ?>>Per Night</option>
                            <option value="per_stay" <?php selected( $service['type'], 'per_stay' ); ?>>Per Stay</option>
                            <option value="per_person" <?php selected( $service['type'], 'per_person' ); ?>>Per Person</option>
                        </select>
                        <button type="button" class="button remove-service">Remove</button>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>
    </div>
    <script>
    jQuery(document).ready(function($) {
        var serviceIndex = <?php echo count( $extra_services ); ?>;
        $('#add-extra-service').on('click', function() {
            var html = '<div class="extra-service-item" style="margin-bottom: 10px; padding: 10px; border: 1px solid #ddd;">' +
                '<input type="text" name="extra_services[' + serviceIndex + '][name]" placeholder="Service Name" style="width: 40%; margin-right: 10px;" />' +
                '<input type="number" name="extra_services[' + serviceIndex + '][price]" placeholder="Price" step="0.01" style="width: 20%; margin-right: 10px;" />' +
                '<select name="extra_services[' + serviceIndex + '][type]" style="width: 20%; margin-right: 10px;">' +
                '<option value="per_night">Per Night</option>' +
                '<option value="per_stay">Per Stay</option>' +
                '<option value="per_person">Per Person</option>' +
                '</select>' +
                '<button type="button" class="button remove-service">Remove</button>' +
                '</div>';
            $('#extra-services-list').append(html);
            serviceIndex++;
        });
        $(document).on('click', '.remove-service', function() {
            $(this).closest('.extra-service-item').remove();
        });
    });
    </script>
    <?php
}

// Save Room Custom Fields
function save_room_custom_fields( $post_id ) {
    // Room Details
    if ( isset( $_POST['room_details_meta_box_nonce'] ) && wp_verify_nonce( $_POST['room_details_meta_box_nonce'], 'room_details_meta_box' ) ) {
        if ( isset( $_POST['room_size'] ) ) {
            update_post_meta( $post_id, '_room_size', sanitize_text_field( $_POST['room_size'] ) );
        }
        if ( isset( $_POST['room_occupancy'] ) ) {
            update_post_meta( $post_id, '_room_occupancy', sanitize_text_field( $_POST['room_occupancy'] ) );
        }
        if ( isset( $_POST['room_max_guests'] ) ) {
            update_post_meta( $post_id, '_room_max_guests', intval( $_POST['room_max_guests'] ) );
        }
        if ( isset( $_POST['room_type'] ) ) {
            update_post_meta( $post_id, '_room_type', sanitize_text_field( $_POST['room_type'] ) );
        }
        if ( isset( $_POST['room_view'] ) ) {
            update_post_meta( $post_id, '_room_view', sanitize_text_field( $_POST['room_view'] ) );
        }
        if ( isset( $_POST['room_bed_type'] ) ) {
            update_post_meta( $post_id, '_room_bed_type', sanitize_text_field( $_POST['room_bed_type'] ) );
        }
        if ( isset( $_POST['room_amenities'] ) ) {
            update_post_meta( $post_id, '_room_amenities', array_map( 'sanitize_text_field', $_POST['room_amenities'] ) );
        } else {
            update_post_meta( $post_id, '_room_amenities', array() );
        }
    }
    
    // Room Pricing
    if ( isset( $_POST['room_pricing_meta_box_nonce'] ) && wp_verify_nonce( $_POST['room_pricing_meta_box_nonce'], 'room_pricing_meta_box' ) ) {
        if ( isset( $_POST['pricing_type'] ) ) {
            update_post_meta( $post_id, '_pricing_type', sanitize_text_field( $_POST['pricing_type'] ) );
        }
        if ( isset( $_POST['price_per_night'] ) ) {
            update_post_meta( $post_id, '_price_per_night', floatval( $_POST['price_per_night'] ) );
        }
        if ( isset( $_POST['price_per_person'] ) ) {
            update_post_meta( $post_id, '_price_per_person', floatval( $_POST['price_per_person'] ) );
        }
        if ( isset( $_POST['deposit_percentage'] ) ) {
            update_post_meta( $post_id, '_deposit_percentage', intval( $_POST['deposit_percentage'] ) );
        }
    }
    
    // Room Availability
    if ( isset( $_POST['room_availability_meta_box_nonce'] ) && wp_verify_nonce( $_POST['room_availability_meta_box_nonce'], 'room_availability_meta_box' ) ) {
        if ( isset( $_POST['min_stay'] ) ) {
            update_post_meta( $post_id, '_min_stay', intval( $_POST['min_stay'] ) );
        }
        if ( isset( $_POST['max_stay'] ) ) {
            update_post_meta( $post_id, '_max_stay', intval( $_POST['max_stay'] ) );
        }
        if ( isset( $_POST['check_in_time'] ) ) {
            update_post_meta( $post_id, '_check_in_time', sanitize_text_field( $_POST['check_in_time'] ) );
        }
        if ( isset( $_POST['check_out_time'] ) ) {
            update_post_meta( $post_id, '_check_out_time', sanitize_text_field( $_POST['check_out_time'] ) );
        }
    }
    
    // Extra Services
    if ( isset( $_POST['room_extra_services_meta_box_nonce'] ) && wp_verify_nonce( $_POST['room_extra_services_meta_box_nonce'], 'room_extra_services_meta_box' ) ) {
        if ( isset( $_POST['extra_services'] ) && is_array( $_POST['extra_services'] ) ) {
            $services = array();
            foreach ( $_POST['extra_services'] as $service ) {
                if ( !empty( $service['name'] ) && !empty( $service['price'] ) ) {
                    $services[] = array(
                        'name' => sanitize_text_field( $service['name'] ),
                        'price' => floatval( $service['price'] ),
                        'type' => sanitize_text_field( $service['type'] ),
                    );
                }
            }
            update_post_meta( $post_id, '_extra_services', $services );
        } else {
            update_post_meta( $post_id, '_extra_services', array() );
        }
    }
}
add_action( 'save_post', 'save_room_custom_fields' );

// ============================================
// 4. BOOKING AVAILABILITY FUNCTIONS
// ============================================

// Get booked dates for a room
function get_room_booked_dates( $room_id ) {
    global $wpdb;
    
    $booked_dates = array();
    
    // Get orders with this product
    $orders = wc_get_orders( array(
        'limit' => -1,
        'status' => array( 'wc-processing', 'wc-on-hold', 'wc-completed' ),
        'return' => 'ids',
    ) );
    
    foreach ( $orders as $order_id ) {
        $order = wc_get_order( $order_id );
        foreach ( $order->get_items() as $item_id => $item ) {
            if ( $item->get_product_id() == $room_id ) {
                $check_in = wc_get_order_item_meta( $item_id, '_check_in_date', true );
                $check_out = wc_get_order_item_meta( $item_id, '_check_out_date', true );
                
                if ( $check_in && $check_out ) {
                    $start = new DateTime( $check_in );
                    $end = new DateTime( $check_out );
                    $end->modify( '-1 day' ); // Check-out date is not booked
                    
                    $current = clone $start;
                    while ( $current <= $end ) {
                        $booked_dates[] = $current->format( 'Y-m-d' );
                        $current->modify( '+1 day' );
                    }
                }
            }
        }
    }
    
    return array_unique( $booked_dates );
}

// Check if room is available for dates
function is_room_available( $room_id, $check_in, $check_out ) {
    $booked_dates = get_room_booked_dates( $room_id );
    
    $start = new DateTime( $check_in );
    $end = new DateTime( $check_out );
    $end->modify( '-1 day' );
    
    $current = clone $start;
    while ( $current <= $end ) {
        $date_str = $current->format( 'Y-m-d' );
        if ( in_array( $date_str, $booked_dates ) ) {
            return false;
        }
        $current->modify( '+1 day' );
    }
    
    return true;
}

// AJAX: Get availability calendar
function ajax_get_room_availability() {
    check_ajax_referer( 'room_booking_nonce', 'nonce' );
    
    $room_id = intval( $_POST['room_id'] );
    $booked_dates = get_room_booked_dates( $room_id );
    
    wp_send_json_success( array(
        'booked_dates' => $booked_dates,
    ) );
}
add_action( 'wp_ajax_get_room_availability', 'ajax_get_room_availability' );
add_action( 'wp_ajax_nopriv_get_room_availability', 'ajax_get_room_availability' );

// AJAX: Calculate booking price
function ajax_calculate_booking_price() {
    check_ajax_referer( 'room_booking_nonce', 'nonce' );
    
    $room_id = intval( $_POST['room_id'] );
    $check_in = sanitize_text_field( $_POST['check_in'] );
    $check_out = sanitize_text_field( $_POST['check_out'] );
    $adults = intval( $_POST['adults'] );
    $children = intval( $_POST['children'] );
    $extra_services = isset( $_POST['extra_services'] ) ? $_POST['extra_services'] : array();
    
    $product = wc_get_product( $room_id );
    if ( !$product ) {
        wp_send_json_error( array( 'message' => 'Room not found' ) );
    }
    
    // Calculate nights
    $check_in_date = new DateTime( $check_in );
    $check_out_date = new DateTime( $check_out );
    $nights = $check_in_date->diff( $check_out_date )->days;
    
    if ( $nights < 1 ) {
        wp_send_json_error( array( 'message' => 'Invalid date range' ) );
    }
    
    // Get pricing
    $pricing_type = get_post_meta( $room_id, '_pricing_type', true );
    $price_per_night = get_post_meta( $room_id, '_price_per_night', true );
    $price_per_person = get_post_meta( $room_id, '_price_per_person', true );
    
    $base_price = 0;
    
    if ( $pricing_type == 'per_night' ) {
        $base_price = floatval( $price_per_night ) * $nights;
    } elseif ( $pricing_type == 'per_person' ) {
        $total_guests = $adults + $children;
        $base_price = floatval( $price_per_person ) * $total_guests * $nights;
    } elseif ( $pricing_type == 'per_night_per_person' ) {
        $total_guests = $adults + $children;
        $base_price = floatval( $price_per_night ) * $nights + ( floatval( $price_per_person ) * $total_guests * $nights );
    } else {
        // Fallback to WooCommerce price
        $base_price = floatval( $product->get_price() ) * $nights;
    }
    
    // Calculate extra services
    $extra_services_price = 0;
    $extra_services_list = get_post_meta( $room_id, '_extra_services', true );
    if ( is_array( $extra_services_list ) && is_array( $extra_services ) ) {
        foreach ( $extra_services as $service_index ) {
            if ( isset( $extra_services_list[ $service_index ] ) ) {
                $service = $extra_services_list[ $service_index ];
                $service_price = floatval( $service['price'] );
                
                if ( $service['type'] == 'per_night' ) {
                    $extra_services_price += $service_price * $nights;
                } elseif ( $service['type'] == 'per_stay' ) {
                    $extra_services_price += $service_price;
                } elseif ( $service['type'] == 'per_person' ) {
                    $total_guests = $adults + $children;
                    $extra_services_price += $service_price * $total_guests;
                }
            }
        }
    }
    
    $total_price = $base_price + $extra_services_price;
    
    // Calculate deposit
    $deposit_percentage = get_post_meta( $room_id, '_deposit_percentage', true );
    if ( empty( $deposit_percentage ) ) {
        $deposit_percentage = 30;
    }
    $deposit_amount = ( $total_price * $deposit_percentage ) / 100;
    $remaining_balance = $total_price - $deposit_amount;
    
    wp_send_json_success( array(
        'nights' => $nights,
        'base_price' => $base_price,
        'extra_services_price' => $extra_services_price,
        'total_price' => $total_price,
        'deposit_percentage' => $deposit_percentage,
        'deposit_amount' => $deposit_amount,
        'remaining_balance' => $remaining_balance,
        'formatted_total' => wc_price( $total_price ),
        'formatted_deposit' => wc_price( $deposit_amount ),
        'formatted_remaining' => wc_price( $remaining_balance ),
    ) );
}
add_action( 'wp_ajax_calculate_booking_price', 'ajax_calculate_booking_price' );
add_action( 'wp_ajax_nopriv_calculate_booking_price', 'ajax_calculate_booking_price' );

// ============================================
// 5. SAVE BOOKING DATA TO ORDER
// ============================================

// Add booking data to cart item
function add_booking_data_to_cart_item( $cart_item_data, $product_id, $variation_id ) {
    if ( isset( $_POST['check_in_date'] ) && isset( $_POST['check_out_date'] ) ) {
        $cart_item_data['check_in_date'] = sanitize_text_field( $_POST['check_in_date'] );
        $cart_item_data['check_out_date'] = sanitize_text_field( $_POST['check_out_date'] );
        $cart_item_data['adults'] = intval( $_POST['adults'] );
        $cart_item_data['children'] = intval( $_POST['children'] );
        if ( isset( $_POST['extra_services'] ) ) {
            $cart_item_data['extra_services'] = array_map( 'intval', $_POST['extra_services'] );
        }
    }
    return $cart_item_data;
}
add_filter( 'woocommerce_add_cart_item_data', 'add_booking_data_to_cart_item', 10, 3 );

// Display booking data in cart
function display_booking_data_in_cart( $item_data, $cart_item ) {
    if ( isset( $cart_item['check_in_date'] ) ) {
        $item_data[] = array(
            'key' => 'Check-in',
            'value' => date_i18n( get_option( 'date_format' ), strtotime( $cart_item['check_in_date'] ) ),
        );
    }
    if ( isset( $cart_item['check_out_date'] ) ) {
        $item_data[] = array(
            'key' => 'Check-out',
            'value' => date_i18n( get_option( 'date_format' ), strtotime( $cart_item['check_out_date'] ) ),
        );
    }
    if ( isset( $cart_item['adults'] ) ) {
        $item_data[] = array(
            'key' => 'Adults',
            'value' => $cart_item['adults'],
        );
    }
    if ( isset( $cart_item['children'] ) ) {
        $item_data[] = array(
            'key' => 'Children',
            'value' => $cart_item['children'],
        );
    }
    return $item_data;
}
add_filter( 'woocommerce_get_item_data', 'display_booking_data_in_cart', 10, 2 );

// Save booking data to order
function save_booking_data_to_order( $item_id, $values, $cart_item_key ) {
    if ( isset( $values['check_in_date'] ) ) {
        wc_add_order_item_meta( $item_id, '_check_in_date', $values['check_in_date'] );
    }
    if ( isset( $values['check_out_date'] ) ) {
        wc_add_order_item_meta( $item_id, '_check_out_date', $values['check_out_date'] );
    }
    if ( isset( $values['adults'] ) ) {
        wc_add_order_item_meta( $item_id, '_adults', $values['adults'] );
    }
    if ( isset( $values['children'] ) ) {
        wc_add_order_item_meta( $item_id, '_children', $values['children'] );
    }
    if ( isset( $values['extra_services'] ) ) {
        wc_add_order_item_meta( $item_id, '_extra_services', $values['extra_services'] );
    }
}
add_action( 'woocommerce_add_order_item_meta', 'save_booking_data_to_order', 10, 3 );

// ============================================
// 6. PARTIAL PAYMENT SUPPORT
// ============================================

// Add deposit as fee to cart
function add_deposit_fee_to_cart() {
    if ( is_admin() && ! defined( 'DOING_AJAX' ) ) {
        return;
    }
    
    $deposit_total = 0;
    
    foreach ( WC()->cart->get_cart() as $cart_item_key => $cart_item ) {
        $product_id = $cart_item['product_id'];
        $deposit_percentage = get_post_meta( $product_id, '_deposit_percentage', true );
        
        if ( !empty( $deposit_percentage ) && isset( $_POST['payment_type'] ) && $_POST['payment_type'] == 'deposit' ) {
            $line_total = $cart_item['line_total'];
            $deposit_amount = ( $line_total * $deposit_percentage ) / 100;
            $deposit_total += $deposit_amount;
        }
    }
    
    if ( $deposit_total > 0 && isset( $_POST['payment_type'] ) && $_POST['payment_type'] == 'deposit' ) {
        WC()->cart->add_fee( 'Deposit', $deposit_total );
    }
}
add_action( 'woocommerce_cart_calculate_fees', 'add_deposit_fee_to_cart' );

// ============================================
// 7. INQUIRY FORM HANDLER
// ============================================

// AJAX: Send room inquiry
function ajax_send_room_inquiry() {
    check_ajax_referer( 'room_booking_nonce', 'nonce' );
    
    $name = sanitize_text_field( $_POST['name'] );
    $email = sanitize_email( $_POST['email'] );
    $message = sanitize_textarea_field( $_POST['message'] );
    $room_id = intval( $_POST['room_id'] );
    
    if ( empty( $name ) || empty( $email ) || empty( $message ) ) {
        wp_send_json_error( array( 'message' => 'Please fill in all fields.' ) );
    }
    
    if ( !is_email( $email ) ) {
        wp_send_json_error( array( 'message' => 'Please enter a valid email address.' ) );
    }
    
    $room_title = get_the_title( $room_id );
    $room_url = get_permalink( $room_id );
    
    // Send email
    $to = get_option( 'admin_email' );
    $subject = sprintf( 'Room Inquiry: %s', $room_title );
    $email_message = sprintf(
        "You have received a new room inquiry.\n\n" .
        "Room: %s\n" .
        "Room URL: %s\n\n" .
        "Customer Name: %s\n" .
        "Customer Email: %s\n\n" .
        "Message:\n%s\n",
        $room_title,
        $room_url,
        $name,
        $email,
        $message
    );
    
    $headers = array(
        'From: ' . $name . ' <' . $email . '>',
        'Reply-To: ' . $email,
    );
    
    $sent = wp_mail( $to, $subject, $email_message, $headers );
    
    if ( $sent ) {
        wp_send_json_success( array( 'message' => 'Your inquiry has been sent successfully!' ) );
    } else {
        wp_send_json_error( array( 'message' => 'Error sending inquiry. Please try again.' ) );
    }
}
add_action( 'wp_ajax_send_room_inquiry', 'ajax_send_room_inquiry' );
add_action( 'wp_ajax_nopriv_send_room_inquiry', 'ajax_send_room_inquiry' );

// ============================================
// 8. VALIDATE BOOKING BEFORE ADD TO CART
// ============================================

// Validate booking availability before adding to cart
function validate_booking_before_add_to_cart( $passed, $product_id, $quantity ) {
    if ( isset( $_POST['check_in_date'] ) && isset( $_POST['check_out_date'] ) ) {
        $check_in = sanitize_text_field( $_POST['check_in_date'] );
        $check_out = sanitize_text_field( $_POST['check_out_date'] );
        
        // Check if dates are valid
        $check_in_date = new DateTime( $check_in );
        $check_out_date = new DateTime( $check_out );
        
        if ( $check_out_date <= $check_in_date ) {
            wc_add_notice( 'Check-out date must be after check-in date.', 'error' );
            return false;
        }
        
        // Check minimum stay
        $min_stay = get_post_meta( $product_id, '_min_stay', true );
        if ( !empty( $min_stay ) ) {
            $nights = $check_in_date->diff( $check_out_date )->days;
            if ( $nights < $min_stay ) {
                wc_add_notice( sprintf( 'Minimum stay is %d nights.', $min_stay ), 'error' );
                return false;
            }
        }
        
        // Check availability
        if ( !is_room_available( $product_id, $check_in, $check_out ) ) {
            wc_add_notice( 'This room is not available for the selected dates. Please choose different dates.', 'error' );
            return false;
        }
        
        // Check max guests
        if ( isset( $_POST['adults'] ) && isset( $_POST['children'] ) ) {
            $adults = intval( $_POST['adults'] );
            $children = intval( $_POST['children'] );
            $max_guests = get_post_meta( $product_id, '_room_max_guests', true );
            
            if ( !empty( $max_guests ) && ( $adults + $children ) > $max_guests ) {
                wc_add_notice( sprintf( 'Maximum %d guests allowed for this room.', $max_guests ), 'error' );
                return false;
            }
        }
    }
    
    return $passed;
}
add_filter( 'woocommerce_add_to_cart_validation', 'validate_booking_before_add_to_cart', 10, 3 );

// ============================================
// 9. CUSTOMIZE PRODUCT ARCHIVE LOOP
// ============================================

// Change "Add to Cart" button text to "View Room" on archive
function change_archive_add_to_cart_text() {
    return 'View Room';
}
add_filter( 'woocommerce_product_add_to_cart_text', 'change_archive_add_to_cart_text' );
add_filter( 'woocommerce_product_single_add_to_cart_text', function() {
    return 'Book Now';
} );

// ============================================
// 10. ADD SWIPER JS FOR GALLERY
// ============================================

function enqueue_swiper_for_rooms() {
    if ( is_product() ) {
        wp_enqueue_style( 'swiper-css', 'https://cdn.jsdelivr.net/npm/swiper@8/swiper-bundle.min.css', array(), '8.0.0' );
        wp_enqueue_script( 'swiper-js', 'https://cdn.jsdelivr.net/npm/swiper@8/swiper-bundle.min.js', array(), '8.0.0', true );
    }
}
add_action( 'wp_enqueue_scripts', 'enqueue_swiper_for_rooms' );

// ============================================
// 11. FLUSH REWRITE RULES ON ACTIVATION
// ============================================

function room_booking_flush_rewrite_rules() {
    change_product_slug_to_rooms( array(), 'product' );
    flush_rewrite_rules();
}
register_activation_hook( __FILE__, 'room_booking_flush_rewrite_rules' );